#include "lf_win.h"

typedef struct select_helper_t
{
  int i;
  int nfds;
  fd_set *readfds;
  fd_set *writefds;
  fd_set *exceptfds;
  int timeout;
  unsigned thrdaddr;
  int ret;
} select_helper_t;

unsigned __stdcall select_helper(void* arg)
{
  select_helper_t *tmp = (select_helper_t*)arg;
  struct timeval tv;
  int i;

  tv.tv_sec = tmp->timeout / 1000;
  tv.tv_usec = (tmp->timeout % 1000) * 1000;
  tmp->ret = select(0, tmp->readfds, tmp->writefds, tmp->exceptfds, &tv);
  if (tmp->ret == SOCKET_ERROR) {
    i = WSAGetLastError();
  }
  return 0;
}

/* timeout is in milliseconds */
int
poll(struct pollfd *ufds, unsigned int nfds, int timeout)
{
  fd_set *readfds;
  fd_set *writefds;
  fd_set *exceptfds;
  unsigned int nfd_sets;
  unsigned int i;
  unsigned int j;
  select_helper_t *tmp;
  HANDLE *thread;
  int ret = 0;
  int rflag;
  int wflag;
  int eflag;

  /* It's only safe to select on MAXIMUM_WAIT_OBJECTS at a time. Find
   * out how many separate selects need to be done. */
  nfd_sets = (nfds + MAXIMUM_WAIT_OBJECTS - 1) / MAXIMUM_WAIT_OBJECTS;

  tmp = (select_helper_t*)malloc(nfd_sets * sizeof (*tmp));
  if (tmp == NULL) {
    goto abort_with_nothing;
  }
  thread = (HANDLE*)malloc(nfd_sets * sizeof (*thread));
  if (thread == NULL) {
    goto abort_with_tmp;
  }
  readfds = (fd_set*)malloc(nfd_sets * sizeof (*readfds));
  if (readfds == NULL) {
    goto abort_with_thread;
  }
  writefds = (fd_set*)malloc(nfd_sets * sizeof (*writefds));
  if (writefds == NULL) {
    goto abort_with_readfds;
  }
  exceptfds = (fd_set*)malloc(nfd_sets * sizeof (*exceptfds));
  if (exceptfds == NULL) {
    goto abort_with_writefds;
  }

  for (i = 0; i < nfd_sets; ++i) {
    FD_ZERO(&readfds[i]);
    FD_ZERO(&writefds[i]);
    FD_ZERO(&exceptfds[i]);
  }

  /* FD_SET modifies fd_count so we don't have to. */
  for (i = 0; i < nfds; ++i) {
    /* initialize revents */
    ufds[i].revents = 0;

    /* set the sockets that we're interested in */
    j = i / MAXIMUM_WAIT_OBJECTS;
    if (ufds[i].events & (POLLIN | POLLPRI)) {
      FD_SET(ufds[i].fd, &readfds[j]);
    }
    if (ufds[i].events & POLLOUT) {
      FD_SET(ufds[i].fd, &writefds[j]);
    }
    FD_SET(ufds[i].fd, &exceptfds[j]);
  }

  /* Do selects in multiple threads. */
  for (i = 0; i < nfd_sets; ++i) {
    tmp[i].i = i;
    tmp[i].readfds = &readfds[i];
    tmp[i].writefds = &writefds[i];
    tmp[i].exceptfds = &exceptfds[i];
    tmp[i].timeout = timeout;
    thread[i] = (HANDLE)_beginthreadex(NULL, 0, select_helper, &tmp[i],
				       CREATE_SUSPENDED, &tmp[i].thrdaddr);
  }
  for (i = 0;i < nfd_sets; ++i) {
    ResumeThread(thread[i]);
  }
  WaitForMultipleObjects(nfd_sets, thread, TRUE, INFINITE);

  for (i = 0; i < nfds; ++i) {
    j = i / MAXIMUM_WAIT_OBJECTS;
    rflag = wflag = eflag = 0;
    if ((ufds[i].events & (POLLIN | POLLPRI)) &&
	FD_ISSET(ufds[i].fd, &readfds[j])) {
      ufds[i].revents |= POLLIN;
      rflag = 1;
    }
    if ((ufds[i].events & POLLOUT) &&
	FD_ISSET(ufds[i].fd, &writefds[j])) {
      ufds[i].revents |= POLLOUT;
      wflag = 1;
    }
    if (FD_ISSET(ufds[i].fd, &exceptfds[j])) {
      ufds[i].revents |= POLLHUP;
      eflag = 1;
    }
    if (rflag || wflag || eflag) {
      ret++;
    }
  }

  free(exceptfds);
 abort_with_writefds:
  free(writefds);
 abort_with_readfds:
  free(readfds);
 abort_with_thread:
  free(thread);
 abort_with_tmp:
  free(tmp);
 abort_with_nothing:
  return ret;

}

/**********************************************************************/
/* pthreads stuff                                                     */
/**********************************************************************/

__declspec(thread) static pthread_t *tls_thread;

void dump_thread(char *str, pthread_t *thread);

static unsigned __stdcall pthread_create_helper(void *arg)
{
  pthread_t *thread;

  thread = arg;

  /* Some pthread functions don't have pthread_t as a parameter. Thread
   * local storage allows us to still get to it. */
  tls_thread = thread;

  thread->ret = thread->start_routine(thread->arg);
  return 0;
}

/**********************************************************************/
/* Thread Routines                                                    */
/**********************************************************************/
/*{{{*/
int pthread_create(pthread_t *thread, const pthread_attr_t *attr,
		   void *(*start_routine)(void *), void *arg)
{
  unsigned thrdaddr;

  assert(thread != NULL);
  assert(attr == NULL);
  assert(start_routine != NULL);

  /* The prototype to _beginthreadex isn't what we'd like. */
  thread->start_routine = start_routine;
  thread->arg = arg;

  thread->cancel = 0;
  /* To allow pass by reference given pass by value. */
  thread->thread = thread;
  thread->handle = (HANDLE)_beginthreadex(NULL, 0, pthread_create_helper,
					  thread, 0, &thrdaddr);
  if (thread->handle == 0) {
    return errno;
  }
  else {
    return 0;
  }
}

int pthread_cancel(pthread_t thread)
{
  /* Hack to get around pass by value. */
  thread.thread->cancel = 1;
  return 0;
}

int pthread_detach(pthread_t thread)
{
  /* We aren't interested in joining but the handle still needs to be
   * closed. It is ok to close a handle to a worker thread before it
   * terminates. It just means that you can't use the handle anymore. */
  CloseHandle(thread.handle);
  return 0;
}

int pthread_equal(pthread_t t1, pthread_t t2)
{
  return t1.handle == t2.handle;
}

void pthread_exit(void *value_ptr)
{
  /* Normally, thread->ret is set in pthread_create_helper. We don' have
   * access to thread here but we did set tls_thread so we use that
   * instead. */
  tls_thread->ret = value_ptr;
  _endthreadex(0);
}

int pthread_join(pthread_t thread, void **value_ptr)
{
  /* Wait for the thread to terminate and then close the handle. */
  WaitForSingleObject(thread.handle, INFINITE);
  CloseHandle(thread.handle);
  if (value_ptr != NULL) {
    *value_ptr = thread.thread->ret;
  }
  return 0;
}

pthread_t pthread_self(void)
{
  return *tls_thread;
}

void pthread_testcancel(void)
{
  if (tls_thread->cancel) {
    pthread_exit(PTHREAD_CANCELED);
  }
}

void pthread_yield(void)
{
  Sleep(0);
}
/*}}}*/

/**********************************************************************/
/* Mutex Routines                                                     */
/**********************************************************************/
/*{{{*/
int pthread_mutex_destroy(pthread_mutex_t *mutex)
{
  assert(mutex != NULL);
  return !CloseHandle(*mutex);
}

int pthread_mutex_init(pthread_mutex_t *mutex,
		       const pthread_mutexattr_t *attr)
{
  assert(mutex != NULL);
  assert(attr == NULL);
  *mutex = CreateMutex(NULL, FALSE, NULL);
  return (*mutex == NULL);
}

int pthread_mutex_lock(pthread_mutex_t *mutex)
{
  assert(mutex != NULL);
  return (WaitForSingleObject(*mutex, INFINITE) != WAIT_FAILED);
}

int pthread_mutex_unlock(pthread_mutex_t *mutex)
{
  assert(mutex != NULL);
  return (ReleaseMutex(*mutex) == 0);
}
/*}}}*/

#if 0
/**********************************************************************/
/* Condition Variable Routines                                        */
/**********************************************************************/
/*{{{*/
int pthread_cond_init(pthread_cond_t *cond,
		      const pthread_condattr_t *attr)
{
  cond->m_gone = 0;
  cond->m_blocked = 0;
  cond->m_waiting = 0;

  cond->m_gate = CreateSemaphore(0, 1, 1, 0);
  cond->m_queue = CreateSemaphore(0, 0, 0x7ffffffflu, 0);
  cond->m_mutex = CreateMutex(0, 0, 0);

  if (!cond->m_gate || !cond->m_queue || !cond->m_mutex) {
    int res = 0;
    if (cond->m_gate) {
      res = CloseHandle(cond->m_gate);
      assert(res);
    }
    if (cond->m_queue) {
      res = CloseHandle(cond->m_queue);
      assert(res);
    }
    if (cond->m_mutex) {
      res = CloseHandle(cond->m_mutex);
      assert(res);
    }
    return EAGAIN;
  }
  return 0;
}

int pthread_cond_destroy(pthread_cond_t *cond)
{
    int res = 0;
    res = CloseHandle(cond->m_gate);
    assert(res);
    res = CloseHandle(cond->m_queue);
    assert(res);
    res = CloseHandle(cond->m_mutex);
    assert(res);
    return 0;
}

int pthread_cond_signal(pthread_cond_t *cond)
{
  unsigned signals = 0;

  int res = 0;
  res = WaitForSingleObject(cond->m_mutex, INFINITE);
  assert(res == WAIT_OBJECT_0);
  
  if (cond->m_waiting != 0) { // the m_gate is already closed
    if (cond->m_blocked == 0) {
      res = ReleaseMutex(cond->m_mutex);
      assert(res);
      return 0;
    }
    
    ++(cond->m_waiting);
    --(cond->m_blocked);
    signals = 1;
  }
  else {
    res = WaitForSingleObject(cond->m_gate, INFINITE);
    assert(res == WAIT_OBJECT_0);
    if (cond->m_blocked > cond->m_gone) {
      if (cond->m_gone != 0) {
	cond->m_blocked -= cond->m_gone;
	cond->m_gone = 0;
      }
      signals = cond->m_waiting = 1;
      --(cond->m_blocked);
    }
    else {
      res = ReleaseSemaphore(cond->m_gate, 1, 0);
      assert(res);
    }
  }
  
  res = ReleaseMutex(cond->m_mutex);
  assert(res);

  if (signals) {
    res = ReleaseSemaphore(cond->m_queue, signals, 0);
    assert(res);
  }
  return 0;
}

int pthread_cond_broadcast(pthread_cond_t *cond)
{
  unsigned signals = 0;
  
  int res = 0;
  res = WaitForSingleObject(cond->m_mutex, INFINITE);
  assert(res == WAIT_OBJECT_0);
  
  if (cond->m_waiting != 0) { // the m_gate is already closed
    if (cond->m_blocked == 0) {
      res = ReleaseMutex(cond->m_mutex);
      assert(res);
      return 0;
    }
    
    cond->m_waiting += (signals = cond->m_blocked);
    cond->m_blocked = 0;
  }
  else {
    res = WaitForSingleObject(cond->m_gate, INFINITE);
    assert(res == WAIT_OBJECT_0);
    if (cond->m_blocked > cond->m_gone) {
      if (cond->m_gone != 0) {
	cond->m_blocked -= cond->m_gone;
	cond->m_gone = 0;
      }
      signals = cond->m_waiting = cond->m_blocked;
      cond->m_blocked = 0;
    }
    else {
      res = ReleaseSemaphore(cond->m_gate, 1, 0);
      assert(res);
    }
  }
  
  res = ReleaseMutex(cond->m_mutex);
  assert(res);
  
  if (signals) {
    res = ReleaseSemaphore(cond->m_queue, signals, 0);
    assert(res);
  }
  return 0;
}

int pthread_cond_wait(pthread_cond_t *cond, pthread_mutex_t *mutex)
{
  int res = 0;
  unsigned was_waiting=0;
  unsigned was_gone=0;

  res = WaitForSingleObject(cond->m_gate, INFINITE);
  assert(res == WAIT_OBJECT_0);
  ++(cond->m_blocked);
  res = ReleaseSemaphore(cond->m_gate, 1, 0);
  assert(res);

  pthread_mutex_unlock(mutex);

  res = WaitForSingleObject(cond->m_queue, INFINITE);
  assert(res == WAIT_OBJECT_0);
  
  res = WaitForSingleObject(cond->m_mutex, INFINITE);
  assert(res == WAIT_OBJECT_0);
  was_waiting = cond->m_waiting;
  was_gone = cond->m_gone;
  if (was_waiting != 0) {
    if (--(cond->m_waiting) == 0) {
      if (cond->m_blocked != 0) {
	res = ReleaseSemaphore(cond->m_gate, 1, 0); // open m_gate
	assert(res);
	was_waiting = 0;
      }
      else if (cond->m_gone != 0)
	cond->m_gone = 0;
    }
  }
  else if (++(cond->m_gone) == (0x7ffffffflu / 2)) {
    // timeout occured, normalize the m_gone count
    // this may occur if many calls to wait with a timeout are made and
    // no call to notify_* is made
    res = WaitForSingleObject(cond->m_gate, INFINITE);
    assert(res == WAIT_OBJECT_0);
    cond->m_blocked -= cond->m_gone;
    res = ReleaseSemaphore(cond->m_gate, 1, 0);
    assert(res);
    cond->m_gone = 0;
  }
  res = ReleaseMutex(cond->m_mutex);
  assert(res);
  
  if (was_waiting == 1) {
    for (/**/ ; was_gone; --was_gone) {
      // better now than spurious later
      res = WaitForSingleObject(cond->m_queue, INFINITE);
      assert(res == WAIT_OBJECT_0);
    }
    res = ReleaseSemaphore(cond->m_gate, 1, 0);
    assert(res);
  }

  pthread_mutex_lock(mutex);

  return 0;
}

#if 0
int pthread_cond_timedwait(pthread_cond_t *cond, pthread_mutex_t *mutex,
			   DWORD milli)
{
  int ret = 0;
  unsigned int res = 0;
  unsigned was_waiting=0;
  unsigned was_gone=0;

  res = WaitForSingleObject(cond->m_gate, INFINITE);
  assert(res == WAIT_OBJECT_0);
  ++(cond->m_blocked);
  res = ReleaseSemaphore(cond->m_gate, 1, 0);
  assert(res);

  pthread_mutex_unlock(mutex);
  
  res = WaitForSingleObject(cond->m_queue, milli);
  assert(res != WAIT_FAILED && res != WAIT_ABANDONED);
  ret = (res == WAIT_OBJECT_0);
  
  res = WaitForSingleObject(cond->m_mutex, INFINITE);
  assert(res == WAIT_OBJECT_0);
  was_waiting = cond->m_waiting;
  was_gone = cond->m_gone;
  if (was_waiting != 0) {
    if (!ret) { // timeout
      if (cond->m_blocked != 0)
	--(cond->m_blocked);
      else
	++(cond->m_gone); // count spurious wakeups
    }
    if (--(cond->m_waiting) == 0) {
      if (cond->m_blocked != 0) {
	res = ReleaseSemaphore(cond->m_gate, 1, 0); // open m_gate
	assert(res);
	was_waiting = 0;
      }
      else if (cond->m_gone != 0)
	cond->m_gone = 0;
    }
  }
  else if (++(cond->m_gone) == (0x7ffffffflu / 2)) {
    // timeout occured, normalize the m_gone count
    // this may occur if many calls to wait with a timeout are made and
    // no call to notify_* is made
    res = WaitForSingleObject(cond->m_gate, INFINITE);
    assert(res == WAIT_OBJECT_0);
    cond->m_blocked -= cond->m_gone;
    res = ReleaseSemaphore(cond->m_gate, 1, 0);
    assert(res);
    cond->m_gone = 0;
  }
  res = ReleaseMutex(cond->m_mutex);
  assert(res);
  
  if (was_waiting == 1) {
    for (/**/ ; was_gone; --was_gone) {
      // better now than spurious later
      res = WaitForSingleObject(cond->m_queue, INFINITE);
      assert(res ==  WAIT_OBJECT_0);
    }
    res = ReleaseSemaphore(cond->m_gate, 1, 0);
    assert(res);
  }

  pthread_mutex_lock(mutex);
  
  if (ret == 0) {
    return ETIMEDOUT;
  }
  else {
    return 0;
  }
}
#endif
/*}}}*/
#endif

/**********************************************************************/
/* Per-Thread Context Routines                                        */
/**********************************************************************/
/*{{{*/
int pthread_key_create(pthread_key_t *key, void (*routine)(void *))
{
  assert(key != NULL);
  assert(routine == NULL);
  *key = TlsAlloc();
  return (*key == 0xFFFFFFFF);
}

int pthread_key_delete(pthread_key_t key)
{
  return !TlsFree(key);
}

void* pthread_getspecific(pthread_key_t key)
{
  return TlsGetValue(key);
}

int pthread_setspecific(pthread_key_t key, const void *value_ptr)
{
  return !TlsSetValue(key, (void*)value_ptr);
}
/*}}}*/
